import type { NextApiRequest, NextApiResponse } from "next";
import {
  sendTaskAssignmentEmail,
  sendTicketAssignmentEmail,
} from "@/lib/email";

/**
 * Test endpoint for email service
 * Usage: POST /api/test-email
 */
export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  if (req.method !== "POST") {
    return res.status(405).json({ ok: false, error: "Method not allowed" });
  }

  const testEmail = "shayan.khalighi76@gmail.com";

  try {
    // Check if API key is set
    if (!process.env.RESEND_API_KEY) {
      return res.status(500).json({
        ok: false,
        error: "RESEND_API_KEY is not set in environment variables",
        message:
          "Please add RESEND_API_KEY to your .env.local file. See EMAIL_SETUP.md for instructions.",
      });
    }

    const results: any[] = [];

    // Test 1: Task assignment email
    try {
      console.log("Sending test task assignment email...");
      await sendTaskAssignmentEmail(testEmail, {
        id: "test-task-id",
        title: "Test Task - Email Service Verification",
        description:
          "This is a test task to verify the email notification system is working correctly. If you received this email, the email service is configured properly!",
        dueDate: new Date(Date.now() + 7 * 24 * 60 * 60 * 1000), // 7 days from now
        assignedBy: {
          name: "Test Admin",
          email: "admin@hospital.com",
        },
      });
      results.push({
        type: "task_assignment",
        status: "sent",
        message: "Task assignment email sent successfully",
      });
    } catch (error: any) {
      results.push({
        type: "task_assignment",
        status: "failed",
        error: error?.message,
      });
    }

    // Test 2: Ticket assignment email
    try {
      console.log("Sending test ticket assignment email...");
      await sendTicketAssignmentEmail(testEmail, {
        id: "test-ticket-id",
        title: "Test Ticket - Email Service Verification",
        description:
          "This is a test ticket to verify the email notification system is working correctly. If you received this email, the email service is configured properly!",
        severity: 2, // High
        department: "IT Department",
        location: "Main Building - Floor 3",
        assignedBy: {
          name: "Test Admin",
          email: "admin@hospital.com",
        },
      });
      results.push({
        type: "ticket_assignment",
        status: "sent",
        message: "Ticket assignment email sent successfully",
      });
    } catch (error: any) {
      results.push({
        type: "ticket_assignment",
        status: "failed",
        error: error?.message,
      });
    }

    const allSent = results.every((r) => r.status === "sent");
    const allFailed = results.every((r) => r.status === "failed");

    return res.status(allSent ? 200 : allFailed ? 500 : 207).json({
      ok: allSent || !allFailed,
      message: allSent
        ? "All test emails sent successfully!"
        : allFailed
        ? "All test emails failed"
        : "Some test emails sent, some failed",
      results,
      recipient: testEmail,
      note: "Please check your inbox (and spam folder) at " + testEmail,
    });
  } catch (error: any) {
    console.error("Error in test email endpoint:", error);
    return res.status(500).json({
      ok: false,
      error: "Failed to send test emails",
      details: error?.message,
    });
  }
}
